/*
TechTech 2001 by Michael Lynn (21/11/2001).
Requirements: 320*200, 8 bit color.

Developed using DirectX 8 runtime and DirectX 8 MiniSDK headers
and libraries using Visual Studio 6 SP5 on Windows 98.
This program will compile using Visual Studio 6 SP6 on Windows 10.

Updated 14/6/2004. Use new framework, load pictures from resources.
Updated 3/1/2002 to fix CreateSurface problem on NT4.0 (DX3) and 98 (DX6).
Updated 13/1/2002. Offscreen surfaces are created in system memory.
Update 4/4/2020. Original sprite graphics used.
Close to original sprite wave-form used.

Initial version: 21/11/2001.

Should be OK with DirectX 4 and higher. Only 10fps on a P200.
*/

#define SZ_NAME		"Surface"
#define SZ_TITLE	"TechTech"

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <windowsx.h>
#include <ddraw.h>

#include <conio.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "resource.h"
#include "ddutil.h"
#include "dxutils.h"

#include "Modfuncs.h"
#include "Npmod32.h"

#define BUFFERS 1		// number of BACK BUFFERS
#define DOUBLEBUFF		// define for the Double Buffered version

char szAppPath[_MAX_PATH+1];

#define width		320
#define height		200

#define RESIZE_WIDTH	width
#define RESIZE_HEIGHT	(height/4)

#define OFFSCREEN_WIDTH		640
#define OFFSCREEN_HEIGHT	400

#define SPRITE_GFX_WIDTH	320
#define SPRITE_GFX_HEIGHT	16

#define VERT_WIDTH		16
#define VERT_HEIGHT		16

LPDIRECTDRAW            lpDD;           // DirectDraw object
LPDIRECTDRAWSURFACE     lpDDSPrimary;   // DirectDraw primary surface
#ifdef DOUBLEBUFF
	LPDIRECTDRAWSURFACE     lpDDSBack;

	LPDIRECTDRAWSURFACE     lpDDSBackRAM;
	LPDIRECTDRAWSURFACE     lpDDSOffOne;      // DirectDraw offscreen surface
	LPDIRECTDRAWSURFACE     lpDDSOffTwo;
	LPDIRECTDRAWSURFACE     lpDDSOffSprites;
	LPDIRECTDRAWSURFACE     lpDDSOffFont;
	LPDIRECTDRAWSURFACE     lpDDSOffFont2;
	LPDIRECTDRAWSURFACE     lpDDSOffFont3;
	LPDIRECTDRAWSURFACE     lpDDSOffFontVert;
	LPDIRECTDRAWSURFACE     lpDDSOffPalette;

	LPDIRECTDRAWSURFACE     lpDDSOffScrollBuffer;
	LPDIRECTDRAWSURFACE     lpDDSOffScrollBuffer2;
	LPDIRECTDRAWSURFACE     lpDDSOffScrollBuffer3;
	LPDIRECTDRAWSURFACE     lpDDSOffScrollResizeBuf;
	LPDIRECTDRAWSURFACE     lpDDSOffScrollVertBuf;
#endif

char buf[256],err=0;
BOOL bActive=TRUE;		// is it faster to count timer count diff in FULLSCREEN mode? 30/6/98?
BOOL gExclusive=TRUE;

int x=0,y=0,dir=0;
int x2=0,y2=0,dir2=0;


#define WITH_SOUND

#define MAX_LINES 30
RECT linePositions[MAX_LINES];
int lineHDirection[MAX_LINES];
int lineVDirection[MAX_LINES];

// scrolling background width and heights
#define back_wid	width
#define back_hei	height
#define MASKCOL		0	// 8 bit paletized, only uses 34 colours so far (0..33) 

#define scr_width	(width-1)
#define scr_height	(height-1)

#define SPRITE_WIDTH	17
#define SPRITE_HEIGHT	10

#define FONT_SCREENWIDTH	width//640
#define FONT_HEIGHT			8

#define FONTVERTSURF_WIDTH	640
#define FONTVERTSHOW_HEIGHT	height//480

#define FONTVERT_HEIGHT		16
#define FONTVERT_WIDTH		16

#define FONTVERTSHOW_WIDTH	16

#define FONT_WIDTH	8
#define FONT_HEIGHT 8

#define outbits	8

unsigned int fc=0;
HPEN gPen;

// prototypes
void init_main(void);
void drawSurface(void);
void drawSurface2(void);
void blitBack(LPDIRECTDRAWSURFACE lpdds);

void cleararea(COLORREF rgb,int left,int top,int right,int bottom);
void clearscreen(COLORREF rgb);
void update_display(void);

void clearSurfaceArea(LPDIRECTDRAWSURFACE lpdds,COLORREF rgb,int left,int top,int right,int bottom);
BOOL LoadPalette( HWND hwnd );
void getpos(char c,int *px,int *py);

void createLines(void);
void stopMusic(void);
BOOL LoadImages( HWND hwnd );
void drawSprites(LPDIRECTDRAWSURFACE lpdds);

void drawScroll1(LPDIRECTDRAWSURFACE lpdds);
void drawScroll2(LPDIRECTDRAWSURFACE lpdds);
void drawScroll3(LPDIRECTDRAWSURFACE lpdds);
void drawVerticalScroll(LPDIRECTDRAWSURFACE lpdds);

int getScrollOffset(char c);
int getScrollOffsetVert(char c);
void setcolor(int c,int r,int g, int b);

// strings and constants
char *szScroll1="WAY BACK IN 1991 WE SAW THE GRODAN AND THE KVACK KVACK DEMO ON THE ATARI ST IN THE SO WATT MEGADEMO BY THE CAREBEARS. IT WAS BASED ON THE STUNNING TECHTECH DEMO FROM THE AMIGA WHICH WAS RELEASED IN 1987. IT NEEDED KICKSTART 1.2 SO I SAW IT FOR THE FIRST TIME IN 2001 ON THE AMIGA EMULATOR. I WANTED TO CONVERT THIS SCREEN TO THE PC TO SEE HOW EFFICIENT DIRECTX IS FOR HANDLING RAW BITMAPS. I HAVE FORGOTTEN HOW BORING IT IS TO WRITE SCROLLTEXTS. I ADMIT THAT THERE ARE BETTER DEMO EFFECTS NOWADAYS BUT THIS STILL HAS A CERTAIN CHARM. CHEERS!                                       ";
char *psz1=szScroll1;

char *szScroll2="NO ONE SAID IT WAS IMPOSSIBLE....      EVEN I KNEW IT WAS POSSIBLE....        SO WE FINALLY CONVERTED THE CLASSIC GRODAN AND THE KVACK KVACK DEMO BY THE CAREBEARS FROM THE ATARI ST TO THE PC. IT WAS BASED ON THE TECHTECH DEMO BY SODAN AND THE MAGICIAN 42 ON THE AMIGA. IT WAS UGLY ON THE AMIGA TOO BUT IT SURE KNOCKED US OFF OUR SEATS WHEN WE SAW IT FOR THE FIRST TIME. WRAP!                                       ";
char *psz2=szScroll2;

char *szScroll3="THE WORD ON THE STREET IS... KEEP IT REAL.                    ";
char *psz3=szScroll3;

char *szScrollVert="OK IT DOES NOT HAVE THE TEDDY BEARS IN THE FONT LIKE THE ATARI ST VERSION. IT JUST SHOWS THAT DIRECTX SHOULD HAVE BETTER RASTER SUPPORT TOO!                    ";
char *pszVert=szScrollVert;
int verty=0,scrollVertXP=0;

char szLoadProblem[_MAX_PATH];

// main display update
DWORD dwFrameTime=0,dwFrames=0,dwFrameCount=0,dwTime;
unsigned long nmax=0;
HWND gHwnd,hwnd2;

// scroll 1
int xoffScroll1Count=0;
int xoffScroll1=0;
int scroll1XP;

// scroll 2
int xoffScroll2Count=0;
int xoffScroll2=0;
int scroll2XP;

// scroll 3
int xoffScroll3Count=0;
int xoffScroll3=0;
int scroll3XP;

// maths package
int spx = 304, spy = 150;
double ychange = 0;
double addy = 0.1;
double siny = 0;
double swing = 0;
double swingy = 0;

void precalc()
{
	int n;
	srand((unsigned)time(NULL));

	// precalc lines
	for(n=0;n<MAX_LINES;n++)
	{
		linePositions[n].top=n*3;
		linePositions[n].left=(width/2)-25;
		linePositions[n].bottom=n*3+8;
		linePositions[n].right=(width/2)+25;
		lineVDirection[n]=1;
		lineHDirection[n]=1;
	}
}

#ifdef WITH_SOUND
void stopMusic()
{
	// stop the music
	StopMod();
	DestroyPlugin();
	AudioExit();
}
#endif

void createLines()
{
	HDC hdc;
	int n;

	for(n=0;n<MAX_LINES;n++)
	{
		if(lineVDirection[n]==1)
		{
			linePositions[n].top+=2;
			linePositions[n].bottom+=1;
			if(linePositions[n].bottom>=height||linePositions[n].top>=height)
				lineVDirection[n]=0;
		}
		else
		{
			linePositions[n].top-=2;
			linePositions[n].bottom-=1;
			if(linePositions[n].bottom<=0||linePositions[n].top<=0)
				lineVDirection[n]=1;
		}

		if(lineHDirection[n]==1)
		{
			linePositions[n].right+=2;
			linePositions[n].left+=1;
			if(linePositions[n].right>=width||linePositions[n].left>=width)
			{
				lineHDirection[n]=0;
			}
		}
		else
		{
			linePositions[n].right-=2;
			linePositions[n].left-=1;
			if(linePositions[n].left<=0||linePositions[n].right<=0)
			{
				lineHDirection[n]=1;
			}
		}
	}
	
	lpDDSBackRAM->GetDC(&hdc);
	SelectObject(hdc,gPen);

	for(n=0;n<MAX_LINES;n++)
	{
		MoveToEx(hdc,linePositions[n].left,linePositions[n].top,NULL);
		LineTo(hdc,linePositions[n].right,linePositions[n].bottom);
	}
	lpDDSBackRAM->ReleaseDC(hdc);
}


void update_display()
{
	//clearscreen(RGB(0,0,0));	
	//setcolor(0,255,0,0);

	// stop incrementing the counter
	// to prevent wrapround issues
	if (fc < 10000) {
		fc++;
	}

	drawSurface();
	
	// time the different effects
	// to start at different points
	if (fc > 600) {
		drawSurface2();
	}
	if (fc > 720) {
		createLines();
	}

	if (fc > 960) {
		drawSprites(lpDDSOffSprites);
	}
	if (fc > 1500) {
		drawVerticalScroll(lpDDSOffFontVert);
	}
	if (fc > 1500) {
		drawScroll1(lpDDSOffFont);
	}
	if (fc > 350) {
		drawScroll2(lpDDSOffFont2);
	}
	if (fc > 2000) {
		drawScroll3(lpDDSOffFont3);
	}

	blitBack(lpDDSBackRAM);
	//setcolor(0,0,0,0);
}

void setcolor(int c,int r,int g, int b)
{
	_outp(0x3c8,c);
	_outp(0x3c9,r);
	_outp(0x3c9,g);
	_outp(0x3c9,b);
}


// do once surface has been setup
void init_main()
{
	clearscreen(RGB(0,0,0));
}

void fillScreen(LPDIRECTDRAWSURFACE lpdds,int x,int y)
{
	RECT r;
	r.top=y; r.bottom=y+height;
	r.left=x; r.right=x+width;
	lpDDSBackRAM->BltFast(0,0,lpdds,&r,DDBLTFAST_WAIT);
}

void fillScreen2(LPDIRECTDRAWSURFACE lpdds,int x,int y)
{
	RECT r;
	r.top=y; r.bottom=y+height;
	r.left=x; r.right=x+width;
	lpDDSBackRAM->BltFast(0,0,lpdds,&r,DDBLTFAST_SRCCOLORKEY|DDBLTFAST_WAIT);
}

void blitBack(LPDIRECTDRAWSURFACE lpdds)
{
	RECT r;
	r.top=0; r.bottom=height;
	r.left=0; r.right=width;
	lpDDSBack->BltFast(0,0,lpdds,&r,DDBLTFAST_WAIT);
}


int xpdiv=0;
void drawSpritesTest(LPDIRECTDRAWSURFACE lpdds)
{
	RECT r;
	int x = 40;
	for(int n=0; n<12; n++)
    {
		r.top = 0;
		r.bottom = SPRITE_HEIGHT;
		r.left = (n * SPRITE_WIDTH);
		r.right = r.left + SPRITE_WIDTH;
            
		lpDDSBackRAM->BltFast(x, 0, lpdds, &r, DDBLTFAST_SRCCOLORKEY|DDBLTFAST_WAIT);
		x += SPRITE_WIDTH;
	}
}

void drawSprite(LPDIRECTDRAWSURFACE lpdds, int n, float dx, float dy)
{
	RECT r;

	r.top = 0;
	r.bottom = SPRITE_HEIGHT;
	r.left = (n * SPRITE_WIDTH);
	r.right = r.left + SPRITE_WIDTH;

	int x = (int)(spx + 290 * cos(swing - dx))/2;
	int y = (int)(spy + ychange * sin(swingy - dy) + siny)/2;

	lpDDSBackRAM->BltFast(x, y, lpdds, &r, DDBLTFAST_SRCCOLORKEY|DDBLTFAST_WAIT);
}

void drawSprites(LPDIRECTDRAWSURFACE lpdds)
{
	if (ychange > 50) {
		addy = -0.1;
	}

	if (ychange < -50) {
		addy = 0.1;
	}

	ychange += addy;
	swing += 0.02;
	swingy += 0.03;
	siny = ychange * sin(swingy);

	float offset = 0;
	for(int n=0; n<12; n++)
	{
		drawSprite(lpdds, n, offset, offset);
		offset += (float)0.2;
	}
}

int getScrollOffset(char c)
{
	if(c>='A'&&c<='Z')
		return ((c-'A')*FONT_WIDTH)+FONT_WIDTH;
	else if(c>='0'&&c<='9')
		return ((c-'0')*FONT_WIDTH)+(30*FONT_WIDTH);
	else if(c=='.')
		return 27*FONT_WIDTH;
	else if(c=='!')
		return 28*FONT_WIDTH;
	else if(c=='?')
		return 29*FONT_WIDTH;
	return 0;
}

int getScrollOffsetVert(char c)
{
	if(c>='A'&&c<='Z')
		return ((c-'A')*FONTVERT_WIDTH)+FONTVERT_WIDTH;
	else if(c>='0'&&c<='9')
		return ((c-'0')*FONTVERT_WIDTH)+(30*FONTVERT_WIDTH);
	else if(c=='.')
		return 27*FONTVERT_WIDTH;
	else if(c=='!')
		return 28*FONTVERT_WIDTH;
	else if(c=='?')
		return 29*FONTVERT_WIDTH;
	return 0;
}

void drawScroll1(LPDIRECTDRAWSURFACE lpdds)
{
	RECT r,rd;

	if(xoffScroll1Count==0)
	{
		char c=*psz1;
		scroll1XP=getScrollOffset(c);
		psz1++;
		if(*psz1==0)
			psz1=szScroll1;
	}

	r.top=0; r.bottom=FONT_HEIGHT-1; 
	r.left=scroll1XP+xoffScroll1Count; r.right=r.left+1;
	rd.top=0; rd.left=width-2;
	rd.right=rd.left+1; rd.bottom=FONT_HEIGHT-1;
	lpDDSOffScrollBuffer->Blt(&rd,lpdds,&r,DDBLT_WAIT,NULL);

	rd.top=0; rd.left=0;
	rd.right=width-1; rd.bottom=FONT_HEIGHT-1;

	r.top=0; r.bottom=FONT_HEIGHT-1;
	r.left=1; r.right=width;
	lpDDSOffScrollBuffer->Blt(&rd,lpDDSOffScrollBuffer,&r,DDBLT_WAIT,NULL);

	r.top=0; r.bottom=FONT_HEIGHT-1;
	r.left=0; r.right=width-1;
	lpDDSBackRAM->BltFast(0,FONT_HEIGHT-1,lpDDSOffScrollBuffer,&r,DDBLTFAST_SRCCOLORKEY|DDBLTFAST_WAIT);

	xoffScroll1Count++;
	if(xoffScroll1Count==FONT_WIDTH) xoffScroll1Count=0;
}

void drawScroll2(LPDIRECTDRAWSURFACE lpdds)
{
	RECT r,rd;

	if(xoffScroll2Count==0)
	{
		char c=*psz2;
		scroll2XP=getScrollOffset(c);
		psz2++;
		if(*psz2==0)
			psz2=szScroll2;
	}

	r.top=0; r.bottom=FONT_HEIGHT-1; 
	r.left=scroll2XP+xoffScroll2Count; r.right=r.left+2;
	rd.top=0; rd.left=width-1-2;
	rd.right=rd.left+2; rd.bottom=FONT_HEIGHT-1;
	lpDDSOffScrollBuffer2->Blt(&rd,lpdds,&r,DDBLT_WAIT,NULL);

	rd.top=0; rd.left=0;
	rd.right=width-2; rd.bottom=FONT_WIDTH-1;

	r.top=0; r.bottom=FONT_WIDTH-1;
	r.left=2; r.right=width;
	lpDDSOffScrollBuffer2->Blt(&rd,lpDDSOffScrollBuffer2,&r,DDBLT_WAIT,NULL);

	r.top=0; r.bottom=FONT_WIDTH-1;
	r.left=0; r.right=width-1;
	lpDDSBackRAM->BltFast(0,FONT_WIDTH+(FONT_WIDTH*2),lpDDSOffScrollBuffer2,&r,DDBLTFAST_SRCCOLORKEY|DDBLTFAST_WAIT);

	xoffScroll2Count+=2;
	if(xoffScroll2Count==FONT_HEIGHT) xoffScroll2Count=0;
}

#define TINYBUFWIDTH	40
void drawScroll3(LPDIRECTDRAWSURFACE lpdds)
{
	RECT r,rd;

	if(xoffScroll3Count==0)
	{
		char c=*psz3;
		scroll3XP=getScrollOffset(c);
		psz3++;
		if(*psz3==0)
			psz3=szScroll3;
	}

	r.top=0; r.bottom=FONT_HEIGHT-1; 
	r.left=scroll3XP+xoffScroll3Count; r.right=r.left+1;
	rd.top=0; rd.left=TINYBUFWIDTH-1;
	rd.right=rd.left+1; rd.bottom=FONT_HEIGHT-1;
	lpDDSOffScrollBuffer3->Blt(&rd,lpdds,&r,DDBLT_WAIT,NULL);

	rd.top=0; rd.left=0;
	rd.right=TINYBUFWIDTH-1;
	rd.bottom=FONT_HEIGHT-1;

	r.top=0; r.bottom=FONT_HEIGHT-1;
	r.left=1; r.right=TINYBUFWIDTH;
	lpDDSOffScrollBuffer3->Blt(&rd,lpDDSOffScrollBuffer3,&r,DDBLT_WAIT,NULL);

	rd.top=0; rd.left=0;
	rd.right=RESIZE_WIDTH; rd.bottom=RESIZE_HEIGHT;
	r.top=0; r.bottom=FONT_HEIGHT-1;
	r.left=0; r.right=TINYBUFWIDTH-2;
	lpDDSOffScrollResizeBuf->Blt(&rd,lpDDSOffScrollBuffer3,&r,DDBLT_WAIT,NULL);
	lpDDSBackRAM->BltFast(0,height-RESIZE_HEIGHT,lpDDSOffScrollResizeBuf,&rd,DDBLTFAST_SRCCOLORKEY|DDBLTFAST_WAIT);

	xoffScroll3Count++;
	if(xoffScroll3Count==FONT_WIDTH) xoffScroll3Count=0;
}

void drawVerticalScroll(LPDIRECTDRAWSURFACE lpdds)
{
	RECT rd,r;

	if(verty==0)
	{
		char c=*pszVert;
		scrollVertXP=getScrollOffsetVert(c);
		pszVert++;
		if(*pszVert==0)
			pszVert=szScrollVert;
		verty=0;
	}
	r.top=verty; r.left=scrollVertXP; r.bottom=r.top+1; r.right=r.left+VERT_HEIGHT;
	lpDDSOffScrollVertBuf->BltFast(0,height-1,lpdds,&r,DDBLTFAST_WAIT);
 
	rd.top=1; rd.left=0; rd.bottom=height; rd.right=VERT_HEIGHT;
	lpDDSOffScrollVertBuf->BltFast(0,0,lpDDSOffScrollVertBuf,&rd,DDBLTFAST_WAIT);
	
	// blit left
	rd.top=0; rd.left=0; rd.bottom=height; rd.right=VERT_HEIGHT;
	lpDDSBackRAM->BltFast(0,0,lpDDSOffScrollVertBuf,&rd,DDBLTFAST_SRCCOLORKEY|DDBLTFAST_WAIT);
	lpDDSBackRAM->BltFast(VERT_WIDTH*2,0,lpDDSOffScrollVertBuf,&rd,DDBLTFAST_SRCCOLORKEY|DDBLTFAST_WAIT);
	lpDDSBackRAM->BltFast(VERT_WIDTH*4,0,lpDDSOffScrollVertBuf,&rd,DDBLTFAST_SRCCOLORKEY|DDBLTFAST_WAIT);

	// blit right
	lpDDSBackRAM->BltFast(width-VERT_WIDTH*5,0,lpDDSOffScrollVertBuf,&rd,DDBLTFAST_SRCCOLORKEY|DDBLTFAST_WAIT);
	lpDDSBackRAM->BltFast(width-VERT_WIDTH*3,0,lpDDSOffScrollVertBuf,&rd,DDBLTFAST_SRCCOLORKEY|DDBLTFAST_WAIT);
	lpDDSBackRAM->BltFast(width-VERT_WIDTH,0,lpDDSOffScrollVertBuf,&rd,DDBLTFAST_SRCCOLORKEY|DDBLTFAST_WAIT);

	verty++;
	if(verty==VERT_HEIGHT)
		verty=0;
}

void drawSurface()
{
	HDC hdcSurf=0;

#ifdef DOUBLEBUFF
//	lpDDSBackRAM->GetDC(&hdcSurf);	// should check rv
#else
	lpDDSPrimary->GetDC(&hdcSurf);	// should check rv
#endif

#ifdef DOUBLEBUFF
	fillScreen(lpDDSOffOne,x,y);
#else
	SetPixel(hdcSurf,xp[x],yp[x],col);
	if(hdcSurf)
		lpDDSPrimary->ReleaseDC(hdcSurf);
#endif

	if(fc>120)
	{
		if(dir==0)
		{
			x+=2; if(x>=width) {x=width; dir=1;}
		}
		else if(dir==1)
		{
			y+=2; if(y>=height) {y=height; dir=2;}
		}
		else if(dir==2)
		{
			x-=2; if(x<=0) {x=0; dir=3;}
		}
		else if(dir==3)
		{
			y-=2; if(y<=0) {y=0; dir=0;}
		}
	}
}

void drawSurface2()
{
	HDC hdcSurf=0;

#ifdef DOUBLEBUFF
//	lpDDSBackRAM->GetDC(&hdcSurf);	// should check rv
#else
	lpDDSPrimary->GetDC(&hdcSurf);	// should check rv
#endif

#ifdef DOUBLEBUFF
	fillScreen2(lpDDSOffTwo,x2,y2);
#else
	SetPixel(hdcSurf,xp[x],yp[x],col);
	if(hdcSurf)
		lpDDSPrimary->ReleaseDC(hdcSurf);
#endif

	if(dir2==0)
	{
		x2+=2; y2+=2;
		if(y2>=height) y2=height;
		if(x2>=width) dir2=1;
	}
	else if(dir2==1)
	{
		x2-=2; if(x2<=0) dir2=2;
	}
	else if(dir2==2)
	{
		x2+=2; y2-=2;
		if(y2<=0) y2=0;
		if(x2>=width) dir2=3;
	}
	else if(dir2==3)
	{
		x2-=2; if(x2<=0) dir2=0;
	}
}


void clearSurfaceArea(LPDIRECTDRAWSURFACE lpdds,COLORREF rgb,int left,int top,int right,int bottom)
{
	DDBLTFX ddbltfx;
	RECT rc;

	ZeroMemory(&ddbltfx,sizeof(ddbltfx));
	ddbltfx.dwSize=sizeof(ddbltfx);
	ddbltfx.dwFillColor=rgb;

	rc.top=top; rc.left=left; rc.bottom=bottom; rc.right=right;
	lpdds->Blt(&rc,NULL,NULL,DDBLT_COLORFILL|DDBLT_WAIT,&ddbltfx);
}

void cleararea(COLORREF rgb,int left,int top,int right,int bottom)
{
	DDBLTFX ddbltfx;
	RECT rc;

	ZeroMemory(&ddbltfx,sizeof(ddbltfx));
	ddbltfx.dwSize=sizeof(ddbltfx);
	ddbltfx.dwFillColor=rgb;

	rc.top=top; rc.left=left; rc.bottom=bottom; rc.right=right;
#ifdef DOUBLEBUFF
	lpDDSBackRAM->Blt(&rc,NULL,NULL,DDBLT_COLORFILL|DDBLT_WAIT,&ddbltfx);
#else
	lpDDSPrimary->Blt(&rc,NULL,NULL,DDBLT_COLORFILL|DDBLT_WAIT,&ddbltfx);
#endif
}

void clearscreen(COLORREF rgb)
{
	DDBLTFX ddbltfx;	
	ZeroMemory(&ddbltfx,sizeof(ddbltfx));
	ddbltfx.dwSize=sizeof(ddbltfx);
	ddbltfx.dwFillColor=rgb;

#ifdef DOUBLEBUFF
	lpDDSBackRAM->Blt(NULL,NULL,NULL,DDBLT_COLORFILL|DDBLT_WAIT,&ddbltfx);
#else
	lpDDSPrimary->Blt(NULL,NULL,NULL,DDBLT_COLORFILL|DDBLT_WAIT,&ddbltfx);
#endif
}

void dumpPalette(HWND hwnd)
{
	LPDIRECTDRAWPALETTE ddp;
	if(FAILED(lpDDSPrimary->GetPalette(&ddp)))
		MessageBox(hwnd,"Get palette failed",SZ_TITLE,MB_OK);

	PALETTEENTRY ape[255];
	ddp->GetEntries(0,0,256,&ape[0]);

	char szBig[1024];
	strcpy(szBig,"{");
	for(int n=0;n<36;n++)
	{
		char szMsg[32];
		sprintf(szMsg,"[%d]->RGB(%d %d %d)",n,ape[n].peRed,ape[n].peGreen,ape[n].peBlue);
		if(n!=36-1)
			strcat(szMsg,", ");
		if(n!=0&&n%10==0)
			strcat(szMsg,"\r\n");
		strcat(szBig,szMsg);
	}
	strcat(szBig,"}");
	MessageBox(hwnd,szBig,"Palette",MB_OK);
}

BOOL LoadPalette( HWND hwnd )
{
	IDirectDrawPalette *p=DDLoadPalette(lpDD,MAKEINTRESOURCE(IDB_PALETTE));
	if(p==NULL)
		return FALSE;

	if(FAILED(lpDDSPrimary->SetPalette( p )))
	{
		MessageBox(hwnd,"Set palette failed",SZ_TITLE,MB_OK);
		return FALSE;
	}
	return true;
}

BOOL LoadImages( HWND hwnd )
{
	HRESULT hr;
	char buf[_MAX_PATH+1];
	dxutils dx;

	szLoadProblem[0]=0;
	DDReLoadBitmap(lpDDSOffPalette,MAKEINTRESOURCE(IDB_PALETTE));

	sprintf(buf,"IDB_CAREBEARS");
	hr=dx.UncompressBitmapFromMemory(lpDDSOffOne,MAKEINTRESOURCE(IDB_CAREBEARS),1);
	if(FAILED(hr)) {strcpy(szLoadProblem,buf); return false;}

	sprintf(buf,"IDB_BACK");
	hr=dx.UncompressBitmapFromMemory(lpDDSOffTwo,MAKEINTRESOURCE(IDB_BACK),2);
	if(FAILED(hr)) {strcpy(szLoadProblem,buf); return false;}

	sprintf(buf,"IDB_SPRITES");
	hr=dx.UncompressBitmapFromMemory(lpDDSOffSprites,MAKEINTRESOURCE(IDB_SPRITES),3);
	if(FAILED(hr)) {strcpy(szLoadProblem,buf); return false;}

	// font for remapping
	sprintf(buf,"IDB_FONT");
	hr=dx.Uncompress4BitBitmapMemory(lpDDSOffFont,MAKEINTRESOURCE(IDB_FONT),4-1,0);
	if(FAILED(hr)) {strcpy(szLoadProblem,buf); return false;}

	sprintf(buf,"IDB_FONT");
	hr=dx.Uncompress4BitBitmapMemory(lpDDSOffFont2,MAKEINTRESOURCE(IDB_FONT),10,0);
	if(FAILED(hr)) {strcpy(szLoadProblem,buf); return false;}

	sprintf(buf,"IDB_FONT");
	hr=dx.Uncompress4BitBitmapMemory(lpDDSOffFont3,MAKEINTRESOURCE(IDB_FONT),10,0);
	if(FAILED(hr)) {strcpy(szLoadProblem,buf); return false;}

	// vertical scroller
	sprintf(buf,"IDB_VERT");
	hr=dx.UncompressBitmapFromMemory(lpDDSOffFontVert,MAKEINTRESOURCE(IDB_VERT),18);
	if(FAILED(hr)) {strcpy(szLoadProblem,buf); return false;}
	return true;
}

static void ReleaseObjects( void )
{
    if ( lpDD != NULL )
    {
        if ( lpDDSPrimary != NULL )
        {
            lpDDSPrimary->Release();
            lpDDSPrimary = NULL;
        }
        lpDD->Release();
        lpDD = NULL;
	}
#ifdef WITH_SOUND
	stopMusic();
#endif

	if(err==1)
	{
		MessageBox(::GetDesktopWindow(),buf,SZ_TITLE,MB_OK);
		err=0;
	}
}

long FAR PASCAL WindowProc( HWND hWnd, UINT message, 
                            WPARAM wParam, LPARAM lParam )
{
    switch ( message )
    {
		case WM_SETCURSOR:
			SetCursor(NULL);	// Turn off the mouse cursor
			return TRUE;

		case WM_KEYDOWN:
			switch ( wParam )
			{
				case VK_ESCAPE:
					SendMessage ( hWnd, WM_CLOSE, 0, 0 );
					break;
			}
			break;

		case WM_DESTROY:
			//setcolor(0,0,0,0);
			ReleaseObjects();
			PostQuitMessage(0);
			break;
		
		default:

			break;
	}
	
    return DefWindowProc( hWnd, message, wParam, lParam );
}

static BOOL doInit( HINSTANCE hInstance, int nCmdShow )
{
    WNDCLASS            wc;
    DDSURFACEDESC       ddsd;
	HRESULT				hr;

	buf[0]=0;
	// Load npmod32.dll
#ifdef WITH_SOUND
	if (!AudioInit())
	{
		MessageBox(NULL,"Unable to load npmod32.dll",SZ_TITLE,MB_OK);
		return FALSE;
	}
#endif
	precalc();

    // Set up and register window class
    wc.style = CS_HREDRAW | CS_VREDRAW;
    wc.lpfnWndProc = WindowProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = hInstance;
    wc.hIcon = LoadIcon( hInstance, IDI_APPLICATION );
    wc.hCursor = LoadCursor( NULL, IDC_ARROW );
    wc.hbrBackground = NULL;
    wc.lpszMenuName = SZ_NAME;
    wc.lpszClassName = SZ_NAME;
    RegisterClass( &wc );

 
    // Create a fullscreen window
    gHwnd = CreateWindowEx(
        WS_EX_TOPMOST,
        SZ_NAME,
        SZ_TITLE,
        WS_POPUP,
        0, 0,
        GetSystemMetrics( SM_CXSCREEN ),
        GetSystemMetrics( SM_CYSCREEN ),
        NULL,
        NULL,
        hInstance,
        NULL );

    hwnd2 = CreateWindowEx(
        WS_EX_TOPMOST,
        SZ_NAME,
        SZ_TITLE,
        WS_POPUP,
        0, 0,
        GetSystemMetrics( SM_CXSCREEN ),
        GetSystemMetrics( SM_CYSCREEN ),
        NULL,
        NULL,
        hInstance,
        NULL );


	// Create the DirectDraw object -- we just need an IDirectDraw
    // interface so we won't bother to query an IDirectDraw2
	hr=DirectDrawCreate( NULL, &lpDD, NULL );
	if(hr!=DD_OK)
	{
		sprintf(buf,"Couldn't create DirectDraw object. Error = 0x%x",hr);
		return FALSE;
	}

	// Set exclusive mode
	hr=lpDD->SetCooperativeLevel( gHwnd, DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN );//| DDSCL_ALLOWREBOOT);
	if(hr!=DD_OK)
	{
		sprintf(buf,"Couldn't set cooperative level. Error = 0x%x",hr);
		return FALSE;
	}

    // Set the display mode.
	hr=lpDD->SetDisplayMode( width, height, outbits );
	if ( FAILED( hr ) )
	{
		sprintf(buf, "Couldn't set display mode. Error = 0x%x",hr);
		return FALSE;
	}

	// check caps
	DDCAPS ddcaps;
	ZeroMemory(&ddcaps,sizeof(ddcaps));
	ddcaps.dwSize = sizeof( ddcaps );
	hr=lpDD->GetCaps( &ddcaps, NULL );
	if ( FAILED( hr ) )
	{
		sprintf(buf, "Couldn't get video card capabilities. Error = 0x%x",hr);
		return FALSE;
    }
    /* // not available in ddraw.h?
	if ( ddcaps.dwCaps & DDCAPS_CANBLTSYSTEM == 0) 
	{
		sprintf(buf, "Cannot blit system memory. Caps are 0x%x. Need 0x%x",ddcaps.dwCaps,DDCAPS_CANBLTSYSTEM);
		return FALSE;
    }
	*/
	// hope this covers it
	if ( ddcaps.dwSVBCaps & DDCAPS_BLT == 0)
	{
		sprintf(buf, "Cannot blit system-to-video memory. Flags are 0x%x. Need 0x%x",ddcaps.dwSVBCaps,DDCAPS_BLT);
		return FALSE;
	}
	// DDCAPS.dwVSBCaps video-to-system memory

	// Create the primary surface
#ifdef DOUBLEBUFF
	ZeroMemory(&ddsd, sizeof(ddsd));
	
	ddsd.dwSize = sizeof( ddsd );
    ddsd.dwFlags = DDSD_CAPS|DDSD_BACKBUFFERCOUNT;
    ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE|DDSCAPS_FLIP|DDSCAPS_COMPLEX|DDSCAPS_VIDEOMEMORY;
	ddsd.dwBackBufferCount = BUFFERS;
	ddsd.dwWidth=width;
	ddsd.dwHeight=height;
    
	// a single call to CreateSurface creates back and front buffers
	// using COMPLEX flag set above! By ML (11-12/7/98) page 123 in book.
	hr = lpDD->CreateSurface( &ddsd, &lpDDSPrimary, NULL );
	if ( FAILED( hr ) )
	{
		sprintf(buf, "Couldn't create primary surface. Error = 0x%x",hr);
		return FALSE;
	}

	ddsd.ddsCaps.dwCaps = DDSCAPS_BACKBUFFER;
	hr = lpDDSPrimary->GetAttachedSurface(&ddsd.ddsCaps,&lpDDSBack);
	if ( FAILED( hr ) )
	{
		sprintf(buf, "Couldn't find the back buffer. Error = 0x%x",hr);
		return FALSE;
	}

#else
	ddsd.dwSize = sizeof( ddsd );
    ddsd.dwFlags = DDSD_CAPS;
    ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;
    
	hr = lpDD->CreateSurface( &ddsd, &lpDDSPrimary, NULL );
	if ( FAILED( hr ) )
	{
		sprintf(buf, "Couldn't create primary surface.");
		return FALSE;
	}
#endif

	// Friday 24/7/98 : Much slower than solid fill blitting!
	ZeroMemory(&ddsd, sizeof(ddsd));
	ddsd.dwSize = sizeof( ddsd );
    ddsd.dwFlags =	DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH | DDSD_CKSRCBLT;
    ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_SYSTEMMEMORY;
    ddsd.dwWidth = OFFSCREEN_WIDTH;
	ddsd.dwHeight = OFFSCREEN_HEIGHT;

	// set mask color for SOURCE blitting
    ddsd.ddckCKSrcBlt.dwColorSpaceLowValue = MASKCOL;
    ddsd.ddckCKSrcBlt.dwColorSpaceHighValue = MASKCOL;

	// Create the 1st offscreen surface.
	hr = lpDD->CreateSurface( &ddsd, &lpDDSOffOne, NULL );
	if ( FAILED( hr ) )
	{
		sprintf(buf,"Couldn't create offscreen surface 1.\nError = 0x%x",hr);
		return FALSE;
	}

	// Create the 2nd offscreen surface.
	hr = lpDD->CreateSurface( &ddsd, &lpDDSOffTwo, NULL );
	if ( FAILED( hr ) )
	{
		sprintf(buf,"Couldn't create offscreen surface 2.\nError = 0x%x",hr);
		return FALSE;
	}

	// create the offscreen sprites
    ddsd.dwWidth = SPRITE_GFX_WIDTH;
	ddsd.dwHeight = SPRITE_GFX_HEIGHT;
	hr = lpDD->CreateSurface( &ddsd, &lpDDSOffSprites, NULL );
	if ( FAILED( hr ) )
	{
		sprintf(buf,"Couldn't create offscreen sprites.\nError = 0x%x",hr);
		return FALSE;
	}

	// create the offscreen font
    ddsd.dwWidth = FONT_SCREENWIDTH;
	ddsd.dwHeight = FONT_HEIGHT;
	hr = lpDD->CreateSurface( &ddsd, &lpDDSOffFont, NULL );
	if ( FAILED( hr ) )
	{
		sprintf(buf,"Couldn't create offscreen font.\nError = 0x%x",hr);
		return FALSE;
	}

	// create the offscreen font
    ddsd.dwWidth = FONT_SCREENWIDTH;
	ddsd.dwHeight = FONT_HEIGHT;
	hr = lpDD->CreateSurface( &ddsd, &lpDDSOffFont2, NULL );
	if ( FAILED( hr ) )
	{
		sprintf(buf,"Couldn't create offscreen font2.\nError = 0x%x",hr);
		return FALSE;
	}

	// create the offscreen font
    ddsd.dwWidth = FONT_SCREENWIDTH;
	ddsd.dwHeight = FONT_HEIGHT;
	hr = lpDD->CreateSurface( &ddsd, &lpDDSOffFont3, NULL );
	if ( FAILED( hr ) )
	{
		sprintf(buf,"Couldn't create offscreen font3.\nError = 0x%x",hr);
		return FALSE;
	}

	// create the offscreen font
    ddsd.dwWidth = FONTVERTSURF_WIDTH;
	ddsd.dwHeight = FONTVERT_HEIGHT;
	hr = lpDD->CreateSurface( &ddsd, &lpDDSOffFontVert, NULL );
	if ( FAILED( hr ) )
	{
		sprintf(buf,"Couldn't create offscreen vertical font.\nError = 0x%x",hr);
		return FALSE;
	}

	// create the offscreen scroll buffer
    ddsd.dwWidth = FONT_SCREENWIDTH;
	ddsd.dwHeight = FONT_HEIGHT;
	hr = lpDD->CreateSurface( &ddsd, &lpDDSOffScrollBuffer, NULL );
	if ( FAILED( hr ) )
	{
		sprintf(buf,"Couldn't create offscreen scroll buffer.\nError = 0x%x",hr);
		return FALSE;
	}
	clearSurfaceArea(lpDDSOffScrollBuffer,RGB(0,0,0),0,0,width,16);

	// create the offscreen scroll buffer 2
    ddsd.dwWidth = FONT_SCREENWIDTH;
	ddsd.dwHeight = FONT_HEIGHT; 
	hr = lpDD->CreateSurface( &ddsd, &lpDDSOffScrollBuffer2, NULL );
	if ( FAILED( hr ) )
	{
		sprintf(buf,"Couldn't create offscreen scroll buffer 2.\nError = 0x%x",hr);
		return FALSE;
	}
	clearSurfaceArea(lpDDSOffScrollBuffer2,RGB(0,0,0),0,0,FONT_SCREENWIDTH,FONT_HEIGHT);

	// create the offscreen scroll buffer 3
    ddsd.dwWidth = FONT_SCREENWIDTH;
	ddsd.dwHeight = FONT_HEIGHT; 
	hr = lpDD->CreateSurface( &ddsd, &lpDDSOffScrollBuffer3, NULL );
	if ( FAILED( hr ) )
	{
		sprintf(buf,"Couldn't create offscreen scroll buffer 3.\nError = 0x%x",hr);
		return FALSE;
	}
	clearSurfaceArea(lpDDSOffScrollBuffer3,RGB(0,0,0),0,0,FONT_SCREENWIDTH,FONT_HEIGHT);

	// create the offscreen scroll resize buffer
    ddsd.dwWidth = RESIZE_WIDTH;
	ddsd.dwHeight = RESIZE_HEIGHT;
	hr = lpDD->CreateSurface( &ddsd, &lpDDSOffScrollResizeBuf, NULL );
	if ( FAILED( hr ) )
	{
		sprintf(buf,"Couldn't create offscreen scroll resize buffer.\nError = 0x%x",hr);
		return FALSE;
	}
	clearSurfaceArea(lpDDSOffScrollResizeBuf,RGB(0,0,0),0,0,RESIZE_WIDTH,RESIZE_HEIGHT);

	// create the offscreen vertical scroll resize buffer
    ddsd.dwWidth = FONTVERTSHOW_WIDTH;
	ddsd.dwHeight = FONTVERTSHOW_HEIGHT;
	hr = lpDD->CreateSurface( &ddsd, &lpDDSOffScrollVertBuf, NULL );
	if ( FAILED( hr ) )
	{
		sprintf(buf,"Couldn't create offscreen vertical scroll resize buffer.\nError = 0x%x",hr);
		return FALSE;
	}
	clearSurfaceArea(lpDDSOffScrollVertBuf,RGB(0,0,0),0,0,FONTVERTSHOW_WIDTH,FONTVERTSHOW_HEIGHT);

	// create the offscreen temporary back buffer,
	// this is blitted to the real back buffer in video RAM in one go
    ddsd.dwWidth = width;
	ddsd.dwHeight = height;
	hr = lpDD->CreateSurface( &ddsd, &lpDDSBackRAM, NULL );
	if ( FAILED( hr ) )
	{
		sprintf(buf,"Couldn't create offscreen temporary back buffer.\nError = 0x%x",hr);
		return FALSE;
	}

	// start
	init_main();
	ShowWindow( gHwnd, nCmdShow );

	// loading palette first doesn't set it on XP, was also unreliable on NT4
    if ( !LoadImages( gHwnd ))
	{
		sprintf(buf,"Couldn't load %s",szLoadProblem);
		return FALSE;
    }

    if ( !LoadPalette( gHwnd ))
	{
		sprintf(buf,"Couldn't load palette.");
		return FALSE;
    }

#ifdef WITH_SOUND
	char buf[_MAX_PATH+1];
	_snprintf(buf,_MAX_PATH,"%s\\ronken.mod",szAppPath);
	PlayMod(buf,hwnd2);
#endif
	return TRUE;
}

int PASCAL WinMain( HINSTANCE hInstance, HINSTANCE hPrevInstance,
                    LPSTR lpCmdLine, int nCmdShow )
{
    MSG         msg;

	if(!_fullpath(&szAppPath[0],".",_MAX_PATH))
	{
		MessageBox(0,"Invalid path",SZ_TITLE,MB_OK);
		return -1;
	}

	gPen=CreatePen(PS_SOLID,1,RGB(255,0,0));
	if(!gPen)
	{
		MessageBox(0,"Cannot create pen",SZ_TITLE,MB_OK);
		return -1;
	}

    lpCmdLine = lpCmdLine;
    hPrevInstance = hPrevInstance;

	err=0;
    if ( !doInit( hInstance, nCmdShow ) )
    {
		err=1; SendMessage(gHwnd,WM_DESTROY,0,0);
        return FALSE;
    }
	
	while( 1 )
    {
		//setcolor(0,255,0,0);

		if( PeekMessage( &msg, NULL, 0, 0, PM_NOREMOVE ) )
		{
			if( !GetMessage( &msg, NULL, 0, 0 ) )
			{
#ifdef WITH_SOUND
				stopMusic();
#endif
				return msg.wParam;
			}
			TranslateMessage(&msg); 
			DispatchMessage(&msg);
		}
		else if( !gExclusive || bActive )
		{

#ifdef DOUBLEBUFF

		update_display();
		//setcolor(0,0,0,0);

		// automatically waits for vbl
		if(FAILED(lpDDSPrimary->Flip(NULL,DDFLIP_WAIT)))
		{
			err=1; sprintf(buf,"Unable to flip video buffers");
			return 1;
		}
#else
		update_display();
#endif

		}
		else
		{
			WaitMessage();
		}
	}

	DeleteObject(gPen);
    return msg.wParam;
}
